/******************************************************************************\
**
**     COMPANY:	LDW Software
**
**      MODULE:	ContentMap
**
** DESCRIPTION:	Maintains the content of the world map, including objects,
**				hotspots, collectables, and blocking information.
**
**      AUTHOR:	Adrian Francis
**
\******************************************************************************/

#ifndef CONTENTMAP_H
#define CONTENTMAP_H

/******************************************************************************\
** LIBRARY INCLUDES
\******************************************************************************/
#include "ldwTypes.h"
#include "../Game/Isola.h"	//EDirection

class CBehaviorEditor;
class CContentMapEditor;
class CWorldPath;

/******************************************************************************\
** CONSTANT DEFINITIONS
\******************************************************************************/

// Block width and height MUST be a power of 2!
const int ContentMap_c_iBlockWidth = 8;
const int ContentMap_c_iBlockHeight = 8;

const int ContentMap_c_iWidthInBlocks = 256;
const int ContentMap_c_iHeightInBlocks = 256;

// A known legal position near the center of the map where something can be
// placed.  Used by FindLegalTerrain.
const ldwPoint ContentMap_c_sSnapToCenter = ldwPoint(795, 1500);

const int ContentMap_c_iMaxObjectCandidates = 1024;

/******************************************************************************\
** CLASS DEFINITIONS
\******************************************************************************/
class CContentMap
{
	friend class CBehaviorEditor;
	friend class CContentMapEditor;
	friend class CWorldPath;

public:

	//class enumerations
	enum EMaterial
	{
		eMaterial_None = 0,
		eMaterial_Soil,
		eMaterial_Grass,
		eMaterial_Sand,
		eMaterial_ShallowWater,
		eMaterial_DeepWater,
		eMaterial_Rock,
		eMaterial_Interior,

		eMaterial_NumberOf
	};

	enum EObject
	{
		eObject_None = 0,
		eObject_Fireplace,
		eObject_Hut,
		eObject_ScienceHut,
		eObject_Vines,
		eObject_Bath,
		eObject_Puddle,

		eObject_AlchemyLab,
		eObject_BrokenLift,
		eObject_RainLiftPlatform,
		eObject_AlchemyPot,
		eObject_AlchemyTable,
		eObject_StatuePath,
		eObject_Firewood,
		eObject_Foodbin,
		eObject_Fire,
		eObject_House1Site,
		eObject_House2Site,
		eObject_House3Site,
		eObject_House1,
		eObject_House2,
		eObject_House3,
		eObject_Tree1,
		eObject_Tree2,
		eObject_Tree3,
		eObject_ResearchTable,
		eObject_ResearchEquipment,
		eObject_AudienceNorth,
		eObject_AudienceCenter,
		eObject_AudienceSouth,
		eObject_Honey,
		eObject_BeeHive,
		eObject_VillageCenter,
		eObject_DiskRoomLid,
		eObject_Ruins,
		eObject_StonePile,
		eObject_SunRoomDoor,
		eObject_KeyJump,
		eObject_ClothingHutSite,
		eObject_HospitalSite,
		eObject_ClothingHut,
		eObject_Hospital,
		eObject_Ocean,
		eObject_Herb6,
		eObject_Herb7,
		eObject_AtFireNW,
		eObject_AtFireSE,
		eObject_DivingPlatform,
		eObject_RosterOfTheDead,
		eObject_LeafPile,
		eObject_AromatherapyHerb,
		eObject_Clam,

		eObject_NumberOf
	};

	enum EHotSpot
	{
		eHotSpot_None = 0,
		eHotSpot_Fireplace,
		eHotSpot_Bath,
		eHotSpot_Firewood,
		eHotSpot_Kindling,
		eHotSpot_Vines,

		eHotSpot_BrokenLift,
		eHotSpot_RainLiftPlatform,
		eHotSpot_Herb1,
		eHotSpot_Herb2,
		eHotSpot_Herb3,
		eHotSpot_Herb4,
		eHotSpot_Herb5,
		eHotSpot_AlchemyPot,
		eHotSpot_AlchemyTable,
		eHotSpot_Foodbin,
		eHotSpot_Amphitheatre,
  		eHotSpot_Beehive,
		eHotSpot_BrokenStatue,
		eHotSpot_StatuePath,
		eHotSpot_Research,
		eHotSpot_House1Site,
		eHotSpot_House2Site,
		eHotSpot_House3Site,
		eHotSpot_FruitTree,
		eHotSpot_SunDisk,
		eHotSpot_Ruins,
		eHotSpot_KeyJump,
		eHotSpot_ClothingHutSite,
		eHotSpot_HospitalSite,
		eHotSpot_Herb6,
		eHotSpot_Herb7,
		eHotSpot_LoveShack,
		eHotSpot_DivingPlatform,
		eHotSpot_RosterOfDead,
		eHotSpot_Chalkboard,
		eHotSpot_BurnedTorch,
		eHotSpot_PathTop,
		eHotSpot_Spring,
		eHotSpot_HospitalLab,
		eHotSpot_Clifftree,

		eHotSpot_NumberOf
	};

	enum ERedirector
	{
		eRedirector_None = 0,
		eRedirector_DivingRock,

		eRedirector_NumberOf
	};

	struct SContent
	{
		bool			m_bIsBlocking;
		EMaterial		m_eMaterial;
		EObject			m_eObject;
		EHotSpot		m_eHotSpot;
		ERedirector		m_eRedirector;
	};

private:

	//class data
	unsigned int	m_uiMap[ContentMap_c_iWidthInBlocks][ContentMap_c_iHeightInBlocks];

	//class methods
	//setup
	public:		CContentMap(void);

	// process
	public:		void Load();
	protected:	const bool Save();
	protected:	void Draw(const bool bShowBlocking, const bool bShowMaterials, const bool bShowObjects, const bool bShowHotSpots, const bool bShowRedirectors, const float fAlpha) const;
	public:		const ldwPoint FindObject(const EObject eObject) const;
	public:		const ldwPoint FindLegalTerrain(const ldwPoint sStartPos, const EDirection eDirectionalBias, const bool bIsDeepWaterLegal) const;
	protected:	inline const bool ReadBlocking(const int iRow, const int iColumn) const;
	protected:	inline const EMaterial ReadMaterial(const int iRow, const int iColumn) const;
	protected:	inline const EObject ReadObject(const int iRow, const int iColumn) const;
	protected:	inline const EHotSpot ReadHotSpot(const int iRow, const int iColumn) const;
	protected:	inline const ERedirector ReadRedirector(const int iRow, const int iColumn) const;
	private:	void Read(int iRow, int iColumn, SContent &sContent) const;
	private:	void Write(const int iRow, const int iColumn, const SContent sContent);
	private:	void ConvertPosToRowColumn(const ldwPoint sPos, int &iRow, int &iColumn) const;
	private:	void ConvertRowColumnToPos(const int iRow, const int iColumn, ldwPoint &sPos) const;

	// access
	protected:	void SetContent(const ldwRect sRect, const bool bIsBlocking, const EMaterial eMaterial = eMaterial_None, const EObject eObject = eObject_None, const EHotSpot eHotSpot = eHotSpot_None, const ERedirector eRedirector = eRedirector_None);
	protected:	void SetContent(const ldwRect sRect, const SContent &sContent);
	protected:	void SetContent(const ldwRect sRect, const SContent &sNewContent, const bool bSetBlocking, const bool bSetMaterial, const bool bSetObject, const bool bSetHotSpot, const bool bSetRedirector);
	protected:	void GetContent(const ldwPoint sPos, SContent &sContent) const;
	public:		const bool IsBlocking(const ldwPoint sPos) const;
	public:		const EMaterial GetMaterial(const ldwPoint sPos) const;
	public:		const EObject GetObject(const ldwPoint sPos) const;
	public:		const EHotSpot GetHotSpot(const ldwPoint sPos) const;
	public:		const ERedirector GetRedirector(const ldwPoint sPos) const;
	public:		void SetObjectBlocking(const EObject eObject, const bool bIsBlocking);
	public:		void SetObjectMaterial(const EObject eObject, const EMaterial eMaterial);
	public:		void SetObjectHotSpot(const EObject eObject, const EHotSpot eHotSpot);
	public:		void SetObjectRedirector(const EObject eObject, const ERedirector eRedirector);
	public:		void SetObjectContent(const EObject eObject, const SContent &sNewContent, const bool bSetBlocking, const bool bSetMaterial, const bool bSetHotSpot, const bool bSetRedirector);
	public:		const bool IsValidSpawnPos(const ldwPoint sPos) const;
};

#include <assert.h>

inline const CContentMap::EMaterial CContentMap::ReadMaterial(const int iRow, const int iColumn) const
{
		assert(iRow >= 0 && iRow < ContentMap_c_iHeightInBlocks &&
		iColumn >= 0 && iColumn < ContentMap_c_iWidthInBlocks);

		return (EMaterial)((m_uiMap[iRow][iColumn] & 0x0000000E) >> 1);
}

inline const bool CContentMap::ReadBlocking(const int iRow, const int iColumn) const
{
	assert(iRow >= 0 && iRow < ContentMap_c_iHeightInBlocks &&
		iColumn >= 0 && iColumn < ContentMap_c_iWidthInBlocks);

	return (m_uiMap[iRow][iColumn] & 0x00000001) ? true : false;
}

/******************************************************************************\
** GLOBAL DECLARATIONS
\******************************************************************************/
extern CContentMap		ContentMap;

#endif

/******************************************************************************\
** END OF FILE
\******************************************************************************/